
/*
 * xm.c - xmodem for mabTerm ver.2.xx
 *
 * 1989-02-??	CRC supported
 * 1990-05-??	for mabTerm ver.2.00 (overlay)
 * 1990-08-??	Bugs of CRC and over 256 blocks are fixed
 *
 * written by Jun Mabuchi
 */

#pragma nonrec

#include <setjmp.h>
#include <string.h>
#include "mtbios.h"
#include "mtbmacro.h"
#include "xmodem.h"

enum trans	{ Ok, End };

#define	EOF		(-1)

#define	OPEN(fname, mode)	mtbios(__OPEN, (int)fname, (int)mode)
#define	CREAT(fname)		mtbios(__CREAT, (int)fname)
#define	CLOSE(fd)		mtbios(__CLOSE, (int)fd)
#define	READ(fd, buf)		mtbios(__READ, (int)fd, (int)buf)
#define	WRITE(fd, buf)		mtbios(__WRITE, (int)fd, (int)buf)

#define	RSPUTC(c)		mtbios(__RSPUTC, (int)c)
#define	RSGETC()		mtbios(__RSGETC)
#define	RSREAD(buf, len)	mtbios(__RSREAD, (int)buf, (int)len)
#define	RSWRITE(buf, len)	mtbios(__RSWRITE, (int)buf, (int)len)
#define	RSRECSIZ()		mtbios(__RSLEN)

/* control data */
#define	SOH	'\x01'		/* start of heading */
#define	STX	'\x02'		/* start of big packet */
#define	ACK	'\x06'		/* acknowledge */
#define	NAK	'\x15'		/* negative acknowledge */
#define	EOT	'\x04'		/* end of transfer */
#define	CAN	'\x18'		/* cancel */
#define	C	'\x43'		/* continue */

/* constants */
#define	XDATASIZE	128	/* data size of XMODEM */
#define	YDATASIZE	1024	/* data size of YMODEM */
#define	SUMPACKSIZE	132	/* size of packet (XMODEM check sum) */
#define	CRCPACKSIZE	133	/* size of packet (XMODEM CRC) */
#define	YPACKSIZE	1029	/* size of packet (YMODEM) */
#define	HANDTIME	60	/* wait time for hand shake (seconds) */
#define	WAITTIME	30	/* wait time for 1 packet (seconds) */
#define	MAXRETRY	20	/* maximum retry times */
#define	TIMEOUT		255


extern long	time(long *);
extern void	sleep(unsigned int);
extern void	dispinit(char *);
extern void	dispratio(int);
extern void	dispdone(void);
extern int	calc_crc(char, int);
extern void	rsatt(void), rsdet(void);


char		crcflg = FALSE;	/* check sum or CRC */
char		fileopen = FALSE;
int		xmfd;
unsigned int	blockNo;
jmp_buf		mainEnv;
unsigned int	packsize = SUMPACKSIZE;
unsigned int	datasize = XDATASIZE;
char		*progname = "";

char		cantOpen[] =  "\r..Can't open \"%s\"\7\n";
char		handShake[] = "\n..Hand shaking";



/*
 *	abort, return to xmodem()
 */
#pragma optimize space
void	myabort(void)
{
	RSPUTC(CAN);	/* send cansel code */
	RSPUTC(CAN);	/* for safety */
	CPUTS("\7\n..Terminate\n");
	longjmp(mainEnv, 1);
}

#pragma optimize space
void	youabort(void)
{
	CPUTS("\7\n..Canceled\n");
	longjmp(mainEnv, 2);
}

#pragma optimize space
void	errabort(void)
{
	RSPUTC(CAN);	/* cansel */
	RSPUTC(CAN);	/* for safety */
	CPUTS("\7\n..Too many errors\n");
	longjmp(mainEnv, 3);
}

/*
 *	check break
 */
#pragma optimize time
void	chkbrk(void)
{
	if (CGETC() == '\03') /* ^C */
		myabort();
}

/*
 *	flush out RS232C receive buffer
 */
#pragma optimize space
void	flsRecBuf(void)
{
	sleep(6);
	while (EOF != RSGETC()) {
		sleep(6);
		chkbrk();
	}
}

/*
 *	receive ACK
 */
#pragma optimize time
char	recvAck(void)
{
	long		timeLimit, timeNow;
	char		c;
	unsigned int	size;

	time(&timeLimit);
	timeLimit += WAITTIME;
	for ( ; timeNow < timeLimit; time(&timeNow)) {
		if ((size = RSRECSIZ()) != 0) {	/* some char is received */
			if (CAN == (c = RSGETC()))
				youabort();
			else
				return (c);
		}
		chkbrk();
	}
	return (TIMEOUT);
}

/*
 *	make a packet to send
 */
#pragma optimize time
void	makePacket(char *src, char *dst)
{
	char		sum;
	unsigned int	crc;
	unsigned int	i;

/* make header */
	*dst++ = SOH;
	*dst++ = (char)blockNo;
	*dst++ = ~((char)blockNo);

/* make data body and calculate check sum or CRC */
	if (crcflg) {
		for (i = 0, crc = 0; i < datasize; ++i) {
			crc = calc_crc(*src, crc);
			*dst++ = *src++;
		}
		*dst++ = 0xff & (crc >> 8);
		*dst   = 0xff & crc;
	}
	else {
		for (i = 0, sum = 0; i < datasize; ++i) {
			sum += *src;
			*dst++ = *src++;
		}
		*dst = sum;
	}
}

/*
 *	send a packet
 */
#pragma optimize time
void	sendPacket(char *data)
{
	char	retry;

	for (retry = MAXRETRY; retry--;) {
		chkbrk();
		RSWRITE(data, packsize);
		switch (recvAck()) {
		case ACK:
			return;
		default:	/* NAK? */
			flsRecBuf();
			break;	/* retry! */
		}
		dispratio(RETRY);
	}
	errabort();
}

/*
 *	wait for C in CRC mode
 */
#pragma optimize time
void	waitC(void)
{
	char	retry;

	for (retry = HANDTIME / WAITTIME; retry--;) {
		switch (recvAck()) {
		case C:
			return;	/* GOOD! start transfer */
		case TIMEOUT:
			break;
		default:	/* error ? */
			flsRecBuf();
			break;
		}
	}
	errabort();
}

/*
 * wait for NAK in SUM mode
 */
#pragma optimize time
void	waitNAK(void)
{
	char	retry;

	for (retry = HANDTIME / WAITTIME; retry--;) {
		switch (recvAck()) {
		case NAK:
			return;	/* GOOD! start transfer */
		case TIMEOUT:
			break;
		default:	/* error ? */
			flsRecBuf();
			break;
		}
	}
	errabort();
}

/*
 *	send a file
 */
#pragma optimize time
void	sendfile(char *fname)
{
	static char	rawbuf[XDATASIZE];
	static char	pakbuf[CRCPACKSIZE];
	unsigned int	len;
	char		retry;
	char		i;

	if (-1 == (xmfd = OPEN(fname, 0))) {
		CPRINTF(cantOpen, fname);
		return;
	}
	fileopen = TRUE;

	/* hand shaking */
	CPUTS(handShake);
	if (crcflg) {
		waitC();
	}
	else {
		waitNAK();
	}
	/* Let's start */
	dispinit(fname);
	for (blockNo = 1; ; ++blockNo) {
		setmem(rawbuf, datasize, '\0');
		if (0 == READ(xmfd, rawbuf))
			break;	/* end of file */
		makePacket(rawbuf, pakbuf);
		sendPacket(pakbuf);
		dispratio(SUCCESS);
	}
	RSPUTC(EOT);	/* end of transfer */
	CLOSE(xmfd);
	fileopen = FALSE;
	/* success? */
	for (i = MAXRETRY; i--; ) {
		if (ACK == recvAck()) {
			dispdone();
			return;		/* all done! */
		}
		else {
			RSPUTC(EOT);
		}
	}
	errabort();
}

/*
 * read 1 packet
 */
#pragma optimize time
enum trans	recvPacket(char *rawbuf)
{
	static char	pakbuf[CRCPACKSIZE];
	unsigned int	i;
	char		sum;
	unsigned int	crc;
	char		retry;
	char		*chp, *bufp;

	for (retry = MAXRETRY; retry--;) {
		switch (recvAck()) {	/* wait SOH, EOT, STX */
		case EOT:	/* end of transfer */
			return (End);

		case SOH:	/* xmodem data or ymodem info block */
			while (packsize - 1 > RSRECSIZ())
				chkbrk();
			RSREAD(pakbuf, packsize - 1);
		/* check block number */
			bufp = rawbuf;
			chp = pakbuf;
			if (*chp != ~*(chp + 1)) { /* block No. lost */
				break;
			}
			if ((0xff & blockNo) != *chp) {/* invalid block No. */
				if (*chp < (0xff & blockNo)) {
					RSPUTC(ACK);	/*already received*/
					retry = MAXRETRY;
					continue;
				}
				break;
			}
		/* copy data and calculate check sum, CRC */
			chp += 2;
			if (crcflg) {
				for (i = 0, crc = 0; i < datasize; ++i) {
					crc = calc_crc(*chp, crc);
					*bufp++ = *chp++;
				}
				if (crc != 
				(((unsigned int)*chp << 8) | *++chp)) {
					break;	/* error, retry */
				}
			}
			else {
				for (i = 0, sum = 0; i < datasize; ++i) {
					sum += *chp;
					*bufp++ = *chp++;
				}
				if (sum != *chp)
					break;	/* error, retry */
			}
			return (Ok);	/* Good! 1 packete received */

/**
		default:
			flsRecBuf();	/* flush out rs buffer */
			break;
**/
		}
		RSPUTC(NAK);	/* request to retry */
		flsRecBuf();
		dispratio(RETRY);
	}
	errabort();	/* too many errors */
}

/*
 *	recieve a file
 */
#pragma optimize time
void	recvfile(char *fname)
{
	static char	rawbuf[XDATASIZE];
	enum trans	stat;

	if (-1 == (xmfd = CREAT(fname))) {
		CPRINTF(cantOpen, fname);
		myabort();	/* send CAN */
	}
	fileopen = TRUE;

/* hand shaking */
	CPUTS(handShake);
	sleep(10 * 60);	/* wait 10 seconds */
	if (crcflg) {
		RSPUTC(C);
	}
	else {
		RSPUTC(NAK);	/* request to send */
	}
	dispinit(fname);
	for (blockNo = 1; ; ++blockNo) {
		if (End == (stat = recvPacket(rawbuf))) { /* end of transfer */
			break;
		}
		if (-1 == WRITE(xmfd, rawbuf)) {
			CPUTS("\n\x07..Disk full\n");
			errabort();
		}
		RSPUTC(ACK);
		dispratio(SUCCESS);
	}
	CLOSE(xmfd);
	fileopen = FALSE;
	RSPUTC(ACK);
	dispdone();
}

#pragma optimize space
void	usage(void)
{
	CPUTS("xmodem.mtc ver.1.01\n");
	CPUTS(progname);
	CPUTS(" {s|r}[c] filename\007\n");
}

/*
 * main of xmodem
 */
#pragma optimize space
#pragma noregalo
void	main(int argc, char **argv)
{
	int	result;

	progname = argv[0];

	if (argc < 3) {
		usage();
		EXIT(1);
	}

	if (argv[1][1] == 'c') {	/* CRC */
		crcflg = TRUE;
		packsize = CRCPACKSIZE;
	}

	rsatt();
	if (0 == (result = setjmp(mainEnv))) {
		if (argv[1][0] == 's')
			sendfile(argv[2]);
		else
			recvfile(argv[2]);
	}
	rsdet();

	if (fileopen)
		CLOSE(xmfd);
	EXIT(0);
}
#pragma regalo
(datasize != _fwrite(rawbuf, datasize, xmfp)) {
			CPUTS("\n\7..Disk full\n");
			errabort();
		}
		RSPUTC(ACK);
		dispratio(SUCCESS);
	}
	dispdone();
	FCLOSE(xmfp);
	fileopen = FALSE;
}

#pragma optimize space
void	usage(void)
{
	CPUTS(progname);
	CPUTS(" {x|y}{s|r}[c] filename\007\n");
}

void	y_env(void)
{
}

void	x_env(void)
{
}

/*
 * main of xmodem/ymodem
 */
#pragma optimize space
#pragma noregalo
void	main(int argc, char **argv)
{
	int	result;

	progname = argv[0];

	if (argc < 3) {
		usage();
		EXIT(1);
	}

	if (argv[1][0] == 'y') {	/* YMODEM */
		yflag = TRUE;
		crcflg = TRUE;
		packsize = YPACKSIZE;
		datasize = YDATASIZE;
	}
	else if (argv[1][2] == 'c') {	/* XMODEM */
		crcflg = TRUE;
		packsize = CRCPACKSIZE;
		datasize = XDATASIZE;
	}

	rsatt();
	if (0 == (result = setjmp(mainEnv))) {
		if (argv[1][1] == 's')
			send(argv[2]);
		else
			recv(argv[2]);
	}
	rsdet();

	if (fileopen)
		FCLOSE(xmfp);
	EXIT(0);
}
#pragma regalo

                                                           